 /**
* \file: FeatureDiscoveryImpl.cpp
*
* \version: 0.1
*
* \release: $Name:$
*
* Includes the internal FeatureDiscovery implementation and
* the communication to DeviceDetector.
*
* \component: Unified SPI
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>

#include <sys/prctl.h>
#include <errno.h>

#include "FeatureDiscoveryImpl.h"

#include "FeatureDiscoveryDefines.h"
#include "Transport.h"


LOG_IMPORT_CONTEXT(uspi_fd)


namespace adit { namespace uspi {


#define USPI_FEATURE_DISCOVERY_EPOLL_MAX_EVENTS    10     ///< The maximum amount of event to report
#define USPI_FEATURE_DISCOVERY_EPOLL_TIME_OUT      500    ///< The maximum time to wait for


FeatureDiscoveryImpl::FeatureDiscoveryImpl(void* inContext, IFeatureDiscoveryCb* inCallbacks, uint32_t inMask)
{
    mContext = inContext;
    mICallbacks = inCallbacks;
    mMask = inMask;
    mImplThreadId = -1;
    mRunning = false;
}

FeatureDiscoveryImpl::~FeatureDiscoveryImpl()
{
    if (mRunning) {
        stop();
    }
}

DiscoveryError FeatureDiscoveryImpl::start()
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr == mICallbacks) {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  IFeatureDiscoveryCb callbacks are not provided", __func__));
        return DiscoveryError::INVALID; /* ====== LEAVE FUNCTION ====== */
    }

    /* thread can execute */
    mRunning = true;

    /* create DeviceDetector */
    mDeviceDetector = std::move(std::unique_ptr<DeviceDetector> (new DeviceDetector(*this)) );

    /* enable the list of device connection types to get notify */
    mDeviceDetector->setMask(mMask);

    /* lock mutex to synchronize creation of ImplThread */
    std::unique_lock<std::mutex> guard(mImplThreadMutex);
    /* create thread */
    int32_t err = pthread_create(&mImplThreadId, nullptr, &ImplThread, this);
    if (0 != err) {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  create ImplThread failed err=%d, errno=%d",
                __func__, err, errno));
        res = DiscoveryError::FAILURE;
    } else {
        /* wait() unlocks mutext and wait for notification */
        mImplThreadCondVar.wait(guard);
        guard.unlock();

        LOG_INFO((uspi_fd, "FeatureDiscoveryImpl::%s()  ImplThread created", __func__));
        res = DiscoveryError::OK;
    }

    return res;
}

DiscoveryError FeatureDiscoveryImpl::stop()
{
    DiscoveryError res = DiscoveryError::OK;

    /* tell thread to stop */
    mRunning = false;

    if (mImplThreadId > 0) {
        int32_t err = pthread_join(mImplThreadId, nullptr);
        LOG_INFO((uspi_fd, "FeatureDiscoveryImpl::%s()  join ImplThread returned with err=%d, errno=%d",
                __func__, err, errno));
        mImplThreadId = -1;
    } else {
        LOG_INFO((uspi_fd, "FeatureDiscoveryImpl::%s()  ImplThread alreay joined or not started", __func__));
    }

    if (mDeviceDetector) {
        mDeviceDetector.reset();
    }

    return res;
}

int32_t FeatureDiscoveryImpl::SetThreadParam(pthread_t inThreadId, std::string inThreadName, int32_t inThreadPrio)
{
    /* set thread name */
    prctl(PR_SET_NAME, inThreadName.c_str(), 0, 0, 0);

    /* set thread priority */
    int err = 0;
    int schedPolicy;
    struct sched_param schedParam;

    err = pthread_getschedparam(inThreadId, &schedPolicy, &schedParam);
    if(err == 0) {
        schedParam.sched_priority = inThreadPrio;
        err = pthread_setschedparam(inThreadId, SCHED_FIFO, &schedParam);
        if(err != 0)
        {
            LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  set priority failed with err=%d, errno=%d",
                    __func__, err, errno));
        }
    } else {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  get priority failed with err=%d, errno=%d",
                __func__, err, errno));
    }

    return err;
}


DiscoveredDeviceUsb::DiscoveredDeviceUsb(const DeviceInfo& inDevInfo, uint32_t inEventMask)
{
    mInfo = inDevInfo;
    mTransport = nullptr;
    mTransport = new Transport<UsbTransport>(mInfo);
    mEventMask = inEventMask;
}

DiscoveredDeviceUsb::~DiscoveredDeviceUsb()
{
    if (nullptr != mTransport) {
        delete mTransport;
        mTransport = nullptr;
    }
}

bool DiscoveredDeviceUsb::checkDevice(SpiFeatureProtocol inProtocol, int32_t inTimeoutMs)
{
    DiscoveryError res = DiscoveryError::FAILURE;
    bool supportProtocol = false;

    uint16_t outTransfered = 0;

    if (nullptr == mTransport) {
        LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Transport was not created!", __func__));

        return supportProtocol; /* ====== LEAVE FUNCTION ====== */
    }

    if ( DiscoveryError::OK != (res = mTransport->open()) ) {
        if (DiscoveryError::FAILURE == res) {
            LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Open device communication failed (res=%s)", __func__, to_str(res)));
        } else {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  Could not open device communication (res=%s)", __func__, to_str(res)));
        }
        return supportProtocol; /* ====== LEAVE FUNCTION ====== */
    }

    switch (inProtocol)
    {
        case FD_PROTOCOL_GOOGLE_AOAP:
        {
            uint8_t DataBuf[2] = {0,0};
            uint16_t DataLength = 2;

            /* send getProtocol request to device */
            res = mTransport->send(REQ_TYPE_USB_DEVICE_TO_HOST, REQ_AOAP_GET_PROTOCOL, \
                                   VALUE_AOAP_GET_PROTOCOL, INDEX_AOAP_GET_PROTOCOL, \
                                   &DataBuf[0], DataLength, &outTransfered, inTimeoutMs);

            if ((DiscoveryError::OK == res) && (outTransfered >= 1)) {
                LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  device supports AOA protocol %d.%d",
                            __func__, DataBuf[0], DataBuf[1]));

                supportProtocol = true;
            } else {
                LOG_INFO((uspi_fd, "FdObjectUsb::%s()  Device does not support control request =%s", __func__, to_str(res)));
            }

            break;
        }
        case FD_PROTOCOL_APPLE_CARPLAY:
        {
            uint8_t DataBuf[4] = {0,0,0,0};
            uint16_t DataLength = 4;

            /* send getProtocol request to device */
            res = mTransport->send(REQ_TYPE_USB_DEVICE_TO_HOST, REQ_APPLE_GET_CAPABILITIES, \
                                   VALUE_APPLE_GET_CAPABILITIES, INDEX_APPLE_GET_CAPABILITIES, \
                                   &DataBuf[0], DataLength, &outTransfered, inTimeoutMs);

            if ((DiscoveryError::OK == res) && (DataBuf[0] == 1)) {
                LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  device supports CarPlay %d %d %d %d",
                            __func__, DataBuf[0], DataBuf[1], DataBuf[2], DataBuf[3]));

                supportProtocol = true;
            } else {
                LOG_INFO((uspi_fd, "FdObjectUsb::%s()  device does not support protocol (%d %d %d %d, res=%s)",
                        __func__, DataBuf[0], DataBuf[1], DataBuf[2], DataBuf[3], to_str(res)));
            }

            break;
        }
        case FD_PROTOCOL_APPLE_NATIVE_HOST_MODE:
        case FD_PROTOCOL_MIRRORLINK:
        {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  There is no possibility to check for support of protocol %d",
                        __func__, inProtocol));
            break;
        }
        default:
        {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  Unsupported protocol %d", __func__, inProtocol));
            break;
        }
    }

    mTransport->close();

    return supportProtocol;
}

DiscoveryError DiscoveredDeviceUsb::switchDevice(SpiFeatureProtocol inProtocol, void* inParam, int32_t inTimeoutMs)
{
    DiscoveryError res = DiscoveryError::FAILURE;

    uint16_t outTransfered = 0;

    if (nullptr == mTransport) {
        LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Transport was not created!", __func__));
        return res; /* ====== LEAVE FUNCTION ====== */
    }

    if ( DiscoveryError::OK != (res = mTransport->open()) ) {
        if (DiscoveryError::FAILURE == res) {
            LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Open device communication failed (res=%s)", __func__, to_str(res)));
        } else {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  Could not open device communication (res=%s)", __func__, to_str(res)));
        }
        return res; /* ====== LEAVE FUNCTION ====== */
    }

    if (0 >= inTimeoutMs)
        inTimeoutMs = 1000;

    switch (inProtocol)
    {
        case FD_PROTOCOL_GOOGLE_AOAP:
        {
            if (nullptr != inParam) {

                LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  Switch to start AOAP", __func__));

                /* call helper API to send all string information to mobile device */
                res = sendStringInformation(inParam, inTimeoutMs);
                if (DiscoveryError::OK == res) {

                    /* send switch command */
                    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_START, \
                                           VALUE_AOAP_START, INDEX_AOAP_START, \
                                           DATA_AOAP_START, DATA_LEN_AOAP_START, &outTransfered, inTimeoutMs);
                } else {
                    LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  send string information failed with res=%s",
                            __func__, to_str(res)));
                }
            } else {
                LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  input parameter is invalid", __func__));
                res = DiscoveryError::INVALID;
            }
            break;
        }
        case FD_PROTOCOL_APPLE_CARPLAY:
        {
            LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  Switch to start CarPlay", __func__));

            res =  mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_APPLE_USB_ROLE_SWITCH, \
                                    VALUE_APPLE_START_CARPLAY, INDEX_APPLE_START_CARPLAY, \
                                    DATA_APPLE_START_CARPLAY, DATA_LEN_APPLE_START_CARPLAY, \
                                    &outTransfered, inTimeoutMs);

            break;
        }
        case FD_PROTOCOL_APPLE_NATIVE_HOST_MODE:
        {
            LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  Switch to start Apple Native Host Mode", __func__));

            res =  mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_APPLE_USB_ROLE_SWITCH, \
                                    VALUE_APPLE_START_NATIVE_HOST_MODE, INDEX_APPLE_START_NATIVE_HOST_MODE, \
                                    DATA_APPLE_START_NATIVE_HOST_MODE, DATA_LEN_APPLE_START_NATIVE_HOST_MODE, \
                                    &outTransfered, inTimeoutMs);

            break;
        }
        case FD_PROTOCOL_MIRRORLINK:
        {
            LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  Switch to start MirrorLink", __func__));

            res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_MIRRORLINK_START, \
                                   VALUE_MIRRORLINK_START, INDEX_MIRRORLINK_START, \
                                   DATA_MIRRORLINK_START, DATA_LEN_MIRRORLINK_START, &outTransfered, inTimeoutMs);

            break;
        }
        default:
        {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  Unknown protocol %d ", __func__, inProtocol));
            res = DiscoveryError::INVALID;
            break;
        }
    }

    LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  Send switch vendor request returned with res=%s",
                __func__, to_str(res)));

    mTransport->close();

    return res;
}

DiscoveryError DiscoveredDeviceUsb::resetDevice(SpiFeatureProtocol inProtocol, int32_t inTimeoutMs)
{
    DiscoveryError res = DiscoveryError::FAILURE;

    (void)inTimeoutMs;
    (void)inProtocol;

    if (nullptr == mTransport) {
        LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Transport was not created!", __func__));
        return res; /* ====== LEAVE FUNCTION ====== */
    }

    if ( DiscoveryError::OK != (res = mTransport->open()) ) {
        if (DiscoveryError::FAILURE == res) {
            LOG_ERROR((uspi_fd, "FdObjectUsb::%s()  Open device communication failed (res=%s)", __func__, to_str(res)));
        } else {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  Could not open device communication (res=%s)", __func__, to_str(res)));
        }
        return res; /* ====== LEAVE FUNCTION ====== */
    }

    switch (inProtocol)
    {
        case FD_PROTOCOL_GOOGLE_AOAP:
        {
            uint32_t maxCnt = 0;
            if (inTimeoutMs > 0) {
                maxCnt = 1;
            } else {
                inTimeoutMs = 1000;
                maxCnt = 10;
            }
            do
            {
                maxCnt--;

                res = mTransport->reset();
                LOGD_DEBUG((uspi_fd, "FdObjectUsb::%s()  reset() returned with %s", __func__, to_str(res)));
                if (DiscoveryError::INCOMPLETE == res) {
                    usleep(inTimeoutMs * 1000);
                }
            } while ((maxCnt > 0) && (DiscoveryError::INCOMPLETE == res));

            if (DiscoveryError::OK == res) {
                LOG_INFO((uspi_fd, "FdObjectUsb::%s()  reset() was successful", __func__));
            } else {
                LOG_WARN((uspi_fd, "FdObjectUsb::%s()  reset() was inconclusive (res: %s)", __func__, to_str(res)));
            }

            break;
        }
        case FD_PROTOCOL_APPLE_CARPLAY:
        case FD_PROTOCOL_MIRRORLINK:
        default:
        {
            LOG_WARN((uspi_fd, "FdObjectUsb::%s()  reset for protocol %d not implemented", __func__, inProtocol));
            res = DiscoveryError::UNSUPPORTED;
            break;
        }
    }

    mTransport->close();

    return res;
}

DiscoveryError DiscoveredDeviceUsb::sendStringInformation(void* inParam, int32_t inTimeoutMs)
{
    DiscoveryError res = DiscoveryError::FAILURE;
    uint16_t outTransfered = 0;

    struct SpiAoapInformation* tmpParam  = (struct SpiAoapInformation*)inParam;

    /* send string information */
    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_MANUFACURER, \
                           (uint8_t*)tmpParam->manufacturer.c_str(), (uint16_t)tmpParam->manufacturer.length(), \
                           &outTransfered, inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_MODEL_NAME, \
                           (uint8_t*)tmpParam->modelName.c_str(), (uint16_t)tmpParam->modelName.length(), \
                           &outTransfered, inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_DESCRIPTION, \
                           (uint8_t*)tmpParam->description.c_str(), (uint16_t)tmpParam->description.length(), \
                           &outTransfered,inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_VERSION, \
                           (uint8_t*)tmpParam->version.c_str(), (uint16_t)tmpParam->version.length(), \
                           &outTransfered,inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_URI, \
                           (uint8_t*)tmpParam->uri.c_str(), (uint16_t)tmpParam->uri.length(), \
                           &outTransfered,inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, REQ_AOAP_STRING_INFORMATION, \
                           VALUE_AOAP_STRING_INFORMATION, INDEX_AOAP_STRING_SERIAL, \
                           (uint8_t*)tmpParam->serial.c_str(), (uint16_t)tmpParam->serial.length(), \
                           &outTransfered,inTimeoutMs);
    if (DiscoveryError::OK != res)
        return res; /* ====== LEAVE FUNCTION ====== */

    /* send audio enabled */
    if (tmpParam->enableAudio > 0) {
        res = mTransport->send(REQ_TYPE_USB_HOST_TO_DEVICE, static_cast<uint8_t> (REQ_AOAP_ENABLE_AUDIO), \
                                1,  //enable audio (2 channel, 16-bit PCM at 44100 Hz)
                                0, 0, 0, &outTransfered, inTimeoutMs);
        if (DiscoveryError::OK != res)
            return res; /* ====== LEAVE FUNCTION ====== */
    }

    return res;
}

void FeatureDiscoveryImpl::handleDeviceEvent(DeviceInfo& devInfo,
        int32_t eventType,
        int32_t eventMask)
{
    /* Only for debug purpose or verbose logging necessary */
//    LOGD_DEBUG((uspi_fd, "FeatureDiscoveryImpl::%s()  called  eventType:%d, eventMask:%d",
//                __func__, eventType, eventMask));
//    devInfo.dump();

    std::shared_ptr<DiscoveredDevice> pDevice(nullptr);

    /* must be adapted in case enum DD_EVENT_MASK will be modified */
    if ((uint32_t)eventMask < DD_BLUETOOTH_OTHER) {  // USB
        LOGD_DEBUG((uspi_fd, "FeatureDiscoveryImpl::%s()  USB eventMask: %d (<%d)", __func__, eventMask, DD_BLUETOOTH_OTHER));
        pDevice = std::make_shared<DiscoveredDeviceUsb>(devInfo, eventMask);
    } else if ((uint32_t)eventMask < DD_ALL) {       // BT
        LOGD_DEBUG((uspi_fd, "FeatureDiscoveryImpl::%s()  BT eventMask: %d (<%d)", __func__, eventMask, DD_ALL));
        pDevice = std::make_shared<DiscoveredDeviceBt>(devInfo, eventMask);
    } else {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Unknown eventMask: %d", __func__, eventMask));
        /* TODO:    anyCb() would provide a nullptr in case eventMask is unknown.
         *          Should we create a DiscoveredDevice object and provide the eventMask as argument? */
    }

    /* call respective callback to inform upper layer */
    switch (eventType)
    {
        case DD_EV_ADD:
        {
            /* notifying about successful switch only for AOAP possible */
            if ((devInfo.getiDeviceInfo(DSYS_IDVENDOR) == 0x18d1)
                && ( (devInfo.getiDeviceInfo(DSYS_IDPRODUCT) >= 0x2d00) && (devInfo.getiDeviceInfo(DSYS_IDPRODUCT) <= 0x2d05) ) ) {
                /* inform upper layer if device appear in AOAP mode */
                mICallbacks->switchedCb(pDevice);
            } else {
                /* any registered device was found. inform upper layer about connection */
                mICallbacks->foundCb(pDevice);
            }
            break;
        }
        case DD_EV_REMOVE:
        {
            /* device disconnection detected */
            mICallbacks->lostCb(pDevice);
            break;
        }
        case DD_EV_CHANGE:
        {
            /* change event received e.g. in case of under voltage */
            mICallbacks->changedCb(pDevice);
            break;
        }
        default:
        {
            LOG_WARN((uspi_fd, "FeatureDiscoveryImpl::%s()  Unknown eventType %d", __func__, eventType));

            /* TODO:    Introduce new and exact error value e.g. "UnknownEventType" */
            mICallbacks->errorCb(DiscoveryError::INVALID);
            break;
        }
    }
}

/// @brief Callback for events
///
/// Executes DeviceDetector::dispatchEvent on event received on
/// DeviceDetector::getEventFd file descriptor.
int32_t FeatureDiscoveryImpl::epollCallback(void* context)
{
    if (nullptr != context) {
        FeatureDiscoveryImpl* me = (FeatureDiscoveryImpl*)context;

        if (me->mDeviceDetector) {
            return me->mDeviceDetector->dispatchEvent();
        }else {
            LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Device Detector not created", __func__));
        }
    } else {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  context is invalid", __func__));
    }
    return -1;
}

/// @brief The registration function.
/// @param efd The epoll file descriptor.
///
/// Register a file descriptor and callback to watch event out.
/// The callback address is copied in the event data pointer, it must therefore
/// be available when the event occurs.
/// @note Code taken from EventHandler::registerEvent method.
int32_t FeatureDiscoveryImpl::epollRegisterEvent(int32_t epollFd)
{
    struct epoll_event event;
    int32_t fd = -1;

    if (!mDeviceDetector)
        return -1; /* ====== LEAVE FUNCTION ====== */

    fd = mDeviceDetector->getEventFd();

    if (fd < 0) {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Wrong parameter to add event fd=%d", __func__, fd));
        return -1; /* ====== LEAVE FUNCTION ====== */
    }
    LOGD_DEBUG((uspi_fd, "FeatureDiscoveryImpl::%s()  Setting up the event handler with fd=%d", __func__, fd));

    event.data.ptr = (void*)FeatureDiscoveryImpl::epollCallback;
    LOGD_DEBUG((uspi_fd, "FeatureDiscoveryImpl::%s()  Providing %p as a callback", __func__, event.data.ptr));

    event.events = EPOLLIN;

    return epoll_ctl(epollFd, EPOLL_CTL_ADD, fd, &event);
}

/// @brief Wait for event on one of the registered file handler.
///
/// The function returns once one of the following happens:
/// - New event
/// - Error
/// - Timeout
///
/// Executed by the @ref main function.
/// On event, the flags are checked, and the callback stored in data.ptr is
/// executed. Error may occurs if the flag are different from (EPOLLIN |
/// EPOLLET), if the callback is not set, or if it returns a negative value.
/// @note The code is taken from the EventHandler::eventWait method.
int32_t FeatureDiscoveryImpl::epollEventWait(int32_t epollFd, struct epoll_event* epollEvents)
{
    int32_t n = 0;
    int32_t res = 0;
    int32_t i = 0;

    if (!epollEvents) {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  No memory available for events", __func__));
        return -1; /* ====== LEAVE FUNCTION ====== */
    }

    n = epoll_wait(epollFd, epollEvents, USPI_FEATURE_DISCOVERY_EPOLL_MAX_EVENTS, USPI_FEATURE_DISCOVERY_EPOLL_TIME_OUT);

    if (n < 0) {
        LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  epoll_wait error: n=%d, %s", __func__, n, strerror(errno)));

        if (errno == EINTR) {
            /* Only exit if the daemon has received QUIT/INT/TERM */
            return 0; /* ====== LEAVE FUNCTION ====== */
        }

        return -1; /* ====== LEAVE FUNCTION ====== */
    } else if (n == 0) {
        LOGD_VERBOSE((uspi_fd, "FeatureDiscoveryImpl::%s()  epoll_wait timed out: n=%d", __func__, n));
    } else {

        for (i = 0 ; i < n ; i++)
        {
            int32_t (*callback)(void *context) = (int32_t (*)(void*))epollEvents[i].data.ptr;

            if (!(epollEvents[i].events & (EPOLLIN | EPOLLET)))
            {
                LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Error while polling. Event received: 0x%X",
                        __func__, epollEvents[i].events));

                /* We only support one event producer.
                 * Error means that this producer died.
                 */
                LOG_INFO((uspi_fd, "FeatureDiscoveryImpl::%s()  Now exiting", __func__));
                res = -1;
                break;
            }

            if (!callback)
            {
                LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Callback not found, exiting", __func__));
                res = -1;
                break;
            }

            if (callback(this) < 0)
            {
                LOG_ERROR((uspi_fd, "FeatureDiscoveryImpl::%s()  Error while calling the callback, exiting", __func__));
                res = -1;
                break;
            }
            else
            {
                LOGD_VERBOSE((uspi_fd, "FeatureDiscoveryImpl::%s()  Callback return successfully",  __func__));
            }
        }
    }

    return res;
}

void* FeatureDiscoveryImpl::ImplThread(void* context) {
    int32_t res = 0;
    int32_t epollFd = -1;

    auto me = static_cast<FeatureDiscoveryImpl*>(context);
    if (me == nullptr) {
        LOG_ERROR((uspi_fd, "ImplThread::%s()  could not cast input pointer", __func__));
        return nullptr; /* ====== LEAVE FUNCTION ====== */
    }
    /* lock mutex to make sure that creator waits for notification */
    std::unique_lock<std::mutex> guard(me->mImplThreadMutex);

    /* set thread attributes */
    me->SetThreadParam(pthread_self(), "FeatureDiscovery", 71);

    epollFd = epoll_create1(0);
    if (0 > epollFd) {
        LOG_ERROR((uspi_fd, "ImplThread::%s()  epoll_create1 error: %d, %s", __func__, epollFd, strerror(errno)));
        return nullptr; /* ====== LEAVE FUNCTION ====== */
    }

    if (0 != me->epollRegisterEvent(epollFd)) {
        LOG_ERROR((uspi_fd, "ImplThread::%s()  could not register epoll event for fd=%d", __func__, epollFd));
        return nullptr; /* ====== LEAVE FUNCTION ====== */
    }

    struct epoll_event* epollEvents = new epoll_event[USPI_FEATURE_DISCOVERY_EPOLL_MAX_EVENTS];
    memset(epollEvents, 0, USPI_FEATURE_DISCOVERY_EPOLL_MAX_EVENTS * sizeof(epoll_event));

    /* unlock mutex and notify creator about successful synchronization */
    guard.unlock();
    me->mImplThreadCondVar.notify_one();

    while ((me->mRunning) && (0 == res))
    {
        res = me->epollEventWait(epollFd, epollEvents);
        if (0 > res) {
            LOG_ERROR((uspi_fd, "ImplThread::%s()  epollEventWait error: %d", __func__, res));
        }
        memset(epollEvents, 0, USPI_FEATURE_DISCOVERY_EPOLL_MAX_EVENTS * sizeof(epoll_event));
    }

    LOG_INFO((uspi_fd, "ImplThread::%s()  stop monitoring for SPI devices", __func__));

    delete[] epollEvents;
    epollEvents = nullptr;

    close(epollFd);

    return nullptr;
}


} } /* namespace adit { namespace uspi { */
